/**
 * Hybrid - Miniapp + Plataform
 * 
 * Lógica:
 * - trc_id na URL → Canal (primeiro acesso)
 * - chat_type === 'channel' + sem trc_id → Broker (retorno)
 * - Caso contrário → Canal (primeiro acesso)
 */
(function () {
  'use strict';

  // Constantes
  const DEFAULTS = {
    DELAY_MIN: 150,
    DELAY_MAX: 300,
    ID_QUERY_KEY: 'site_id',
    LOG_LIMIT: 200
  };

  // Debug overlay (apenas quando necessário)
  (function initDebug() {
    try {
      const search = (location.search || '').replace(/amp%3B/gi, '').replace(/amp;/gi, '');
      const params = new URLSearchParams(search);
      const isDebug = params.get('debug') === '1' || localStorage.getItem('trc:hybrid_debug') === '1';
      const isVerbose = params.get('verbose') === '1';
      
      if (!isDebug) return;

      const panel = document.getElementById('trcDbgPanel');
      const out = document.getElementById('trcDbgOut');
      if (!panel || !out) return;

      let seq = 0;
      const buf = [];
      
      function append(text) {
        try {
          const ts = new Date().toISOString();
          out.textContent += `\n[${ts}] ${text}`;
          out.scrollTop = out.scrollHeight;
        } catch {}
      }

      window.__trc_log = function(entry) {
        try {
          const text = typeof entry === 'string' ? entry : JSON.stringify(entry);
          if (panel.style.display === 'block') {
            append(text);
          } else {
            buf.push(text);
          }
        } catch {}
      };

      panel.style.display = 'block';

      // Controles de debug
      try {
        const bar = document.createElement('div');
        bar.style.cssText = 'position:sticky;top:0;display:flex;gap:8px;margin-bottom:8px;background:rgba(0,0,0,0.9);padding:4px';
        
        ['Copiar', 'Baixar', 'Limpar'].forEach((label, i) => {
          const btn = document.createElement('button');
          btn.textContent = label;
          btn.onclick = i === 0 ? () => {
            try { navigator.clipboard.writeText(out.textContent || ''); } catch {}
          } : i === 1 ? () => {
            try {
              const blob = new Blob([out.textContent || ''], { type: 'text/plain' });
              const url = URL.createObjectURL(blob);
              const a = document.createElement('a');
              a.href = url;
              a.download = 'hybrid-logs.txt';
              a.click();
              setTimeout(() => URL.revokeObjectURL(url), 1000);
            } catch {}
          } : () => { 
            out.textContent = '';
            try { localStorage.removeItem('trc:logs:hybrid'); } catch {}
          };
          bar.appendChild(btn);
        });
        
        panel.insertBefore(bar, out);
      } catch {}

      buf.forEach(append);
      buf.length = 0;
      append(`debug ativado (${search || 'no_qs'})`);
      if (isVerbose) append('verbose=1');

      // Error handlers
      window.addEventListener('error', (ev) => {
        try {
          window.__trc_log({ kind: 'error', message: ev.message, file: ev.filename, line: ev.lineno });
        } catch {}
      });
      window.addEventListener('unhandledrejection', (ev) => {
        try {
          window.__trc_log({ kind: 'unhandled', reason: String(ev.reason) });
        } catch {}
      });
    } catch {}
  })();

  // Logger estruturado (apenas em debug)
  function log(kind, data) {
    if (!window.__trc_log) return;
    try {
      const payload = { kind, seq: (window.__trc_seq = (window.__trc_seq || 0) + 1), t: Math.round(performance.now()) };
      if (data) Object.assign(payload, data);
      window.__trc_log(payload);
      logLocal(payload);
    } catch {}
  }

  // Log local (apenas em debug)
  const DEBUG_MODE = (function() {
    try {
      const search = (location.search || '').replace(/amp%3B/gi, '').replace(/amp;/gi, '');
      const params = new URLSearchParams(search);
      return params.get('debug') === '1' || localStorage.getItem('trc:hybrid_debug') === '1';
    } catch {
      return false;
    }
  })();

  function logLocal(entry) {
    if (!DEBUG_MODE) return;
    try {
      const now = new Date().toISOString();
      const arr = JSON.parse(localStorage.getItem('trc:logs:hybrid') || '[]');
      arr.push(Object.assign({ ts: now }, entry || {}));
      if (arr.length > DEFAULTS.LOG_LIMIT) arr.splice(0, arr.length - DEFAULTS.LOG_LIMIT);
      localStorage.setItem('trc:logs:hybrid', JSON.stringify(arr));
    } catch {}
  }

  // Cache de params/hash (usado múltiplas vezes)
  const cachedParams = (function() {
    try {
      const search = (location.search || '').replace(/amp%3B/gi, '').replace(/amp;/gi, '');
      return new URLSearchParams(search);
    } catch {
      return new URLSearchParams();
    }
  })();

  const cachedHash = (function() {
    try {
      const hash = location.hash || '';
      return hash.startsWith('#') ? hash.substring(1) : hash;
    } catch {
      return '';
    }
  })();

  const cachedHashParams = (function() {
    try {
      if (!cachedHash) return null;
      return new URLSearchParams(cachedHash);
    } catch {
      return null;
    }
  })();

  // Carrega configuração
  async function loadConfig() {
    try {
      if (window.__trc_config_ready && typeof window.__trc_config_ready.then === 'function') {
        await window.__trc_config_ready;
      }
      
      const cfg = window.__getTrcConfig?.() || {};
      const links = cfg.links || {};
      const redirect = cfg.redirect || {};
      const storage = cfg.storage || {};
      
      return {
        channelLink: links.channel_fallback || null,
        brokerUrl: links.plataform_default || links.broker_default || null,
        idQueryKey: redirect.id_query_key || DEFAULTS.ID_QUERY_KEY,
        delayMin: redirect.delay_ms_min || DEFAULTS.DELAY_MIN,
        delayMax: redirect.delay_ms_max || DEFAULTS.DELAY_MAX,
        redirectEnabled: redirect.enabled !== false,
        allowQueryOverride: redirect.allow_query_override !== false,
        holdParam: redirect.hold_param || 'hold',
        customLinkKey: storage.custom_link_key || 'custom_link'
      };
    } catch {
      return {
        channelLink: null,
        brokerUrl: null,
        idQueryKey: DEFAULTS.ID_QUERY_KEY,
        delayMin: DEFAULTS.DELAY_MIN,
        delayMax: DEFAULTS.DELAY_MAX,
        redirectEnabled: true,
        allowQueryOverride: true,
        holdParam: 'hold',
        customLinkKey: 'custom_link'
      };
    }
  }

  // Detecta se trc_id está presente na URL
  function hasTrcIdInUrl() {
    // 1) Query params
    try {
      const qsTrcId = cachedParams.get('startapp') || cachedParams.get('start_param') || cachedParams.get('tgWebAppStartParam');
      if (qsTrcId) return true;
    } catch {}

    // 2) Hash params
    try {
      if (cachedHashParams) {
        const tgData = cachedHashParams.get('tgWebAppData');
        if (tgData) {
          const inner = new URLSearchParams(tgData);
          if (inner.get('start_param')) return true;
        }
      }
    } catch {}

    // 3) Telegram initData
    try {
      const initData = window.Telegram?.WebApp?.initData;
      if (initData) {
        const parts = initData.split('&');
        for (const part of parts) {
          const [key, value] = part.split('=');
          if (key === 'start_param' && value) return true;
        }
      }
    } catch {}

    // 4) Função compartilhada (pode ter start_param congelado)
    try {
      const tgId = window.__trc_telegram?.getTrackingId?.();
      if (tgId && tgId !== 'unknown') {
        // Verificar se veio da URL ou do storage
        const fromStorage = localStorage.getItem('trc:tracking_id');
        if (tgId !== fromStorage) return true; // Se diferente do storage, veio da URL
      }
    } catch {}

    return false;
  }

  // Extrai trc_id da URL (prioridade) ou storage
  function getTrcId() {
    // 1) Query params
    try {
      const qsTrcId = cachedParams.get('startapp') || cachedParams.get('start_param') || cachedParams.get('tgWebAppStartParam');
      if (qsTrcId) return qsTrcId;
    } catch {}

    // 2) Hash params
    try {
      if (cachedHashParams) {
        const tgData = cachedHashParams.get('tgWebAppData');
        if (tgData) {
          const inner = new URLSearchParams(tgData);
          const startParam = inner.get('start_param');
          if (startParam) return startParam;
        }
      }
    } catch {}

    // 3) Telegram initData
    try {
      const initData = window.Telegram?.WebApp?.initData;
      if (initData) {
        const parts = initData.split('&');
        for (const part of parts) {
          const [key, value] = part.split('=');
          if (key === 'start_param' && value) return decodeURIComponent(value);
        }
      }
    } catch {}

    // 4) Função compartilhada
    try {
      const tgId = window.__trc_telegram?.getTrackingId?.();
      if (tgId && tgId !== 'unknown') return tgId;
    } catch {}

    // 5) Storage (fallback)
    try {
      return localStorage.getItem('trc:tracking_id') || localStorage.getItem('tracking_id');
    } catch {}

    return null;
  }

  // Extrai user_id do Telegram
  function getUserId() {
    try {
      const unsafe = window.Telegram?.WebApp?.initDataUnsafe;
      if (unsafe?.user?.id) return String(unsafe.user.id);
    } catch {}

    try {
      const initData = window.Telegram?.WebApp?.initData;
      if (initData) {
        const parts = initData.split('&');
        for (const part of parts) {
          const [key, value] = part.split('=');
          if (key === 'user' && value) {
            const user = JSON.parse(decodeURIComponent(value));
            if (user?.id) return String(user.id);
          }
        }
      }
    } catch {}

    try {
      if (cachedHashParams) {
        const tgData = cachedHashParams.get('tgWebAppData');
        if (tgData) {
          const inner = new URLSearchParams(tgData);
          const userRaw = inner.get('user');
          if (userRaw) {
            const user = JSON.parse(decodeURIComponent(userRaw));
            if (user?.id) return String(user.id);
          }
        }
      }
    } catch {}

    try {
      return localStorage.getItem('trc:user_id') || localStorage.getItem('user_id');
    } catch {}

    return null;
  }

  // Obtém chat_type
  function getChatType() {
    try {
      if (window.__TRC_TG?.chat_type) return String(window.__TRC_TG.chat_type);
    } catch {}

    try {
      const unsafe = window.Telegram?.WebApp?.initDataUnsafe;
      if (unsafe?.chat_type) return String(unsafe.chat_type);
    } catch {}

    try {
      const initData = window.Telegram?.WebApp?.initData;
      if (initData) {
        const parts = initData.split('&');
        for (const part of parts) {
          const [key, value] = part.split('=');
          if (key === 'chat_type' && value) return decodeURIComponent(value);
        }
      }
    } catch {}

    try {
      if (cachedHashParams) {
        const tgData = cachedHashParams.get('tgWebAppData');
        if (tgData) {
          const inner = new URLSearchParams(tgData);
          const ct = inner.get('chat_type');
          if (ct) return ct;
        }
      }
    } catch {}

    try {
      const ct = window.__trc_telegram?.getChatType?.();
      if (ct) return ct;
    } catch {}

    return undefined;
  }

  // Salva dados no localStorage
  function saveData(userId, trcId, link) {
    try {
      if (userId) localStorage.setItem('trc:user_id', userId);
      if (trcId) {
        localStorage.setItem('trc:trc_id', trcId);
        localStorage.setItem('trc:tracking_id', trcId);
      }
      if (link) localStorage.setItem('trc:hybrid_link', link);
      localStorage.setItem('trc:hybrid_first_access_at', String(Date.now()));
      return true;
    } catch {
      return false;
    }
  }

  // Constrói URL do broker com trc_id
  function buildBrokerUrl(brokerUrl, trcId, idQueryKey) {
    if (!brokerUrl || !trcId) return null;

    try {
      const url = new URL(brokerUrl);
      
      // Detecta se tem afftrack= na URL original
      const hasAfftrack = url.searchParams.has('afftrack') || brokerUrl.includes('afftrack=');
      const paramKey = hasAfftrack ? 'afftrack' : idQueryKey;
      
      url.searchParams.set(paramKey, trcId);
      url.searchParams.set('_cb', String(Date.now()));
      
      return url.toString();
    } catch {
      return null;
    }
  }

  // Primeiro acesso: OpenApp → Canal
  function handleFirstAccess(userId, trcId, config, delay) {
    log('first_access_start', { userId: !!userId, trcId: !!trcId });

    // Selecionar destino
    const customLink = localStorage.getItem(`trc:${config.customLinkKey}`) || localStorage.getItem(config.customLinkKey);
    const destino = customLink || config.channelLink;

    if (!destino) {
      log('error', { message: 'Nenhum destino configurado' });
      return;
    }

    // Configuração de redirect
    const allowQuery = config.allowQueryOverride;
    const hold = allowQuery && cachedParams.has(config.holdParam);
    const debugMode = cachedParams.get('debug') === '1';
    let redirectEnabled = config.redirectEnabled;
    if (debugMode) redirectEnabled = false;

    if (!redirectEnabled || hold) {
      log('no_redirect', { reason: hold ? 'hold' : 'debug' });
      return;
    }

    // Salvar dados
    saveData(userId, trcId, destino);
    log('data_saved', { userId: !!userId, trcId: !!trcId });

    // Redirecionar para canal
    const minDelay = Math.max(config.delayMin, isFinite(delay) ? Math.min(delay, config.delayMax) : config.delayMin);
    setTimeout(() => {
      log('redirect', { to: 'channel', url: destino });
      try { location.replace(destino); } catch { location.href = destino; }
    }, minDelay);
  }

  // Retorno: OpenPlataform → Broker (OTIMIZADO para redirect rápido)
  function handleReturn(trcId, config, delay) {
    log('return_start', { trcId: !!trcId });

    if (!trcId) {
      log('error', { message: 'trc_id ausente para retorno' });
      return;
    }

    // Construir URL do broker
    const brokerUrl = buildBrokerUrl(config.brokerUrl, trcId, config.idQueryKey);
    if (!brokerUrl) {
      log('error', { message: 'brokerUrl não configurado ou inválido' });
      return;
    }
    log('broker_url_built', { url: brokerUrl });

    // Configuração de redirect
    const allowQuery = config.allowQueryOverride;
    const hold = allowQuery && cachedParams.has(config.holdParam);
    const debugMode = cachedParams.get('debug') === '1';
    let redirectEnabled = config.redirectEnabled;
    if (debugMode) redirectEnabled = false;

    if (!redirectEnabled || hold) {
      log('no_redirect', { reason: hold ? 'hold' : 'debug' });
      return;
    }

    // Delay mínimo reduzido para broker (mais agressivo)
    const minDelay = Math.max(0, isFinite(delay) ? Math.min(delay, config.delayMax) : Math.min(config.delayMin, 100));

    // Redirecionar para broker
    setTimeout(() => {
      log('redirect', { to: 'broker', url: brokerUrl });
      try { 
        location.assign(brokerUrl); 
      } catch { 
        location.href = brokerUrl; 
      }
    }, minDelay);
  }

  // Função principal
  async function main() {
    try {
      // Limpar logs se solicitado
      if (cachedParams.get('logs') === 'clear') {
        try { localStorage.removeItem('trc:logs:hybrid'); } catch {}
      }

      const delay = parseInt(cachedParams.get('delay') || '0', 10);
      log('start', { href: location.href });

      // Telegram ready
      try {
        window.Telegram?.WebApp?.ready();
        log('tg_ready');
      } catch {}

      // Carregar config
      const config = await loadConfig();
      log('config_loaded', { 
        channelLink: !!config.channelLink, 
        brokerUrl: !!config.brokerUrl,
        idQueryKey: config.idQueryKey 
      });

      // Extrair dados
      const userId = getUserId();
      const trcId = getTrcId();
      const chatType = getChatType();
      const hasTrcIdInUrlFlag = hasTrcIdInUrl();

      log('data_extracted', { userId: !!userId, trcId: !!trcId, chatType, hasTrcIdInUrl: hasTrcIdInUrlFlag });

      // Decisão de fluxo
      if (hasTrcIdInUrlFlag) {
        // trc_id na URL → Primeiro acesso (sobrepõe chat_type)
        log('flow_decision', { flow: 'first_access', reason: 'trc_id_in_url' });
        handleFirstAccess(userId, trcId, config, delay);
      } else if (chatType === 'channel') {
        // chat_type === 'channel' + sem trc_id → Retorno
        log('flow_decision', { flow: 'return', reason: 'channel_no_trc_id' });
        handleReturn(trcId, config, delay);
      } else {
        // Qualquer outro caso → Primeiro acesso
        log('flow_decision', { flow: 'first_access', reason: 'default' });
        handleFirstAccess(userId, trcId, config, delay);
      }
    } catch (error) {
      log('fatal_error', { error: String(error), stack: error?.stack });
    }
  }

  // Iniciar
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', main);
  } else {
    main();
  }
})();
