/*
  Ferramenta de limpeza de dados do domínio atual
  - Lista e limpa: localStorage, sessionStorage, cookies, CacheStorage, IndexedDB, Service Workers
  - Exibe estimativa de uso via navigator.storage.estimate()
  Notas de segurança:
  - Operações são "best-effort"; ambientes WebView (ex.: Telegram) podem limitar APIs
  - Cookies com atributos HttpOnly/SameSite/Path/Domain específicos podem não ser removidos pelo JS
  - IndexedDB: não há API padronizada para enumerar bancos; usamos heurísticas
*/

(function () {
  const $ = (id) => document.getElementById(id);
  const updatedAtEl = $('updatedAt');
  const envInfoEl = $('envInfo');
  const storageSummaryEl = $('storageSummary');
  const diag = {
    raw: document.getElementById('diagTgRaw'),
    json: document.getElementById('diagTgJson'),
    copyRawBtn: document.getElementById('copyDiagTgRawBtn'),
    clearRawBtn: document.getElementById('clearDiagTgRawBtn'),
    copyJsonBtn: document.getElementById('copyDiagTgJsonBtn'),
  };

  const el = {
    local: {
      count: $('localStorageCount'),
      table: $('localStorageTable'),
      error: $('localStorageError'),
      clearBtn: $('clearLocalStorageBtn'),
    },
    session: {
      count: $('sessionStorageCount'),
      table: $('sessionStorageTable'),
      error: $('sessionStorageError'),
      clearBtn: $('clearSessionStorageBtn'),
    },
    cookies: {
      count: $('cookiesCount'),
      table: $('cookiesTable'),
      error: $('cookiesError'),
      clearBtn: $('clearCookiesBtn'),
    },
    caches: {
      count: $('cachesCount'),
      table: $('cachesTable'),
      error: $('cachesError'),
      clearBtn: $('clearCachesBtn'),
    },
    idb: {
      count: $('idbCount'),
      table: $('idbTable'),
      error: $('idbError'),
      clearBtn: $('clearIdbBtn'),
    },
    sw: {
      count: $('swCount'),
      table: $('swTable'),
      error: $('swError'),
      clearBtn: $('clearSwBtn'),
    },
    refreshBtn: $('refreshBtn'),
    reloadBtn: $('reloadBtn'),
    clearAllBtn: $('clearAllBtn'),
    confirmClearAllBtn: $('confirmClearAllBtn'),
  };

  function setUpdatedNow() {
    const now = new Date();
    updatedAtEl.textContent = `Atualizado em ${now.toLocaleString()}`;
  }

  function setEnvInfo() {
    try {
      const ua = navigator.userAgent;
      const origin = location.origin;
      envInfoEl.textContent = `Origem: ${origin}\nUA: ${ua}`;
    } catch (e) {
      // ignore
    }
  }

  function createActionButton(label, handler, extraClass) {
    const btn = document.createElement('button');
    if (extraClass) btn.className = extraClass;
    btn.textContent = label;
    btn.addEventListener('click', handler);
    return btn;
  }

  function renderTableRow(cells) {
    const tr = document.createElement('tr');
    for (const cell of cells) {
      const td = document.createElement('td');
      if (cell instanceof Node) {
        td.appendChild(cell);
      } else {
        td.textContent = String(cell);
      }
      tr.appendChild(td);
    }
    return tr;
  }

  function safeJSON(value) {
    try { return JSON.stringify(JSON.parse(value), null, 2); } catch { return value; }
  }

  // localStorage
  function listLocalStorage() {
    el.local.error.style.display = 'none';
    el.local.table.innerHTML = '';
    try {
      const keys = [];
      for (let i = 0; i < localStorage.length; i++) keys.push(localStorage.key(i));
      keys.sort((a, b) => String(a).localeCompare(String(b)));
      for (const key of keys) {
        const raw = localStorage.getItem(key);
        const value = safeJSON(raw);
        const pre = document.createElement('pre');
        pre.className = 'kbd form-control wrap-anywhere';
        pre.textContent = value;
        const removeBtn = createActionButton('Remover', () => {
          try { localStorage.removeItem(key); } finally { listLocalStorage(); updateEstimate(); }
        }, 'danger');
        el.local.table.appendChild(renderTableRow([key, pre, removeBtn]));
      }
      el.local.count.textContent = `(${keys.length})`;
    } catch (err) {
      el.local.error.style.display = '';
      el.local.error.textContent = 'Erro ao acessar localStorage: ' + (err && err.message ? err.message : String(err));
      el.local.count.textContent = '(erro)';
    }
  }

  function clearLocalStorage() {
    try {
      localStorage.clear();
    } catch {}
    listLocalStorage();
    updateEstimate();
  }

  // sessionStorage
  function listSessionStorage() {
    el.session.error.style.display = 'none';
    el.session.table.innerHTML = '';
    try {
      const keys = [];
      for (let i = 0; i < sessionStorage.length; i++) keys.push(sessionStorage.key(i));
      keys.sort((a, b) => String(a).localeCompare(String(b)));
      for (const key of keys) {
        const raw = sessionStorage.getItem(key);
        const value = safeJSON(raw);
        const pre = document.createElement('pre');
        pre.className = 'kbd form-control wrap-anywhere';
        pre.textContent = value;
        const removeBtn = createActionButton('Remover', () => {
          try { sessionStorage.removeItem(key); } finally { listSessionStorage(); updateEstimate(); }
        }, 'danger');
        el.session.table.appendChild(renderTableRow([key, pre, removeBtn]));
      }
      el.session.count.textContent = `(${keys.length})`;
    } catch (err) {
      el.session.error.style.display = '';
      el.session.error.textContent = 'Erro ao acessar sessionStorage: ' + (err && err.message ? err.message : String(err));
      el.session.count.textContent = '(erro)';
    }
  }

  function clearSessionStorage() {
    try { sessionStorage.clear(); } catch {}
    listSessionStorage();
    updateEstimate();
  }

  // Cookies
  function parseCookies() {
    const map = [];
    try {
      const raw = document.cookie || '';
      if (!raw) return map;
      const parts = raw.split(';');
      for (const part of parts) {
        const [name, ...rest] = part.trim().split('=');
        map.push({ name, value: rest.join('=') });
      }
    } catch {}
    return map;
  }

  function listCookies() {
    el.cookies.error.style.display = 'none';
    el.cookies.table.innerHTML = '';
    try {
      const cookies = parseCookies();
      cookies.sort((a, b) => a.name.localeCompare(b.name));
      for (const c of cookies) {
        const valuePre = document.createElement('pre');
        valuePre.className = 'kbd';
        valuePre.textContent = c.value;
        const removeBtn = createActionButton('Remover', () => {
          try {
            // Melhor esforço: define expiração passada e paths comuns
            const base = `${encodeURIComponent(c.name)}=; expires=Thu, 01 Jan 1970 00:00:00 GMT;`;
            const pathAttrs = ['','/'];
            const domains = deriveCookieDomains(location.hostname);
            for (const path of pathAttrs) {
              document.cookie = `${base} path=${path}`;
              for (const d of domains) {
                document.cookie = `${base} path=${path}; domain=${d}`;
              }
            }
          } catch {}
          listCookies();
          updateEstimate();
        }, 'danger');
        el.cookies.table.appendChild(renderTableRow([c.name, valuePre, removeBtn]));
      }
      el.cookies.count.textContent = `(${cookies.length})`;
    } catch (err) {
      el.cookies.error.style.display = '';
      el.cookies.error.textContent = 'Erro ao acessar cookies: ' + (err && err.message ? err.message : String(err));
      el.cookies.count.textContent = '(erro)';
    }
  }

  function deriveCookieDomains(hostname) {
    // Gera variações de domínio para tentar excluir cookies em diferentes níveis
    const parts = hostname.split('.');
    const domains = [];
    for (let i = 0; i < parts.length - 1; i++) {
      domains.push(parts.slice(i).join('.'));
    }
    return domains;
  }

  function clearAllCookies() {
    const cookies = parseCookies();
    try {
      const baseExpiry = 'Thu, 01 Jan 1970 00:00:00 GMT';
      const domains = deriveCookieDomains(location.hostname);
      const paths = ['','/'];
      for (const cookie of cookies) {
        for (const path of paths) {
          document.cookie = `${encodeURIComponent(cookie.name)}=; expires=${baseExpiry}; path=${path}`;
          for (const d of domains) {
            document.cookie = `${encodeURIComponent(cookie.name)}=; expires=${baseExpiry}; path=${path}; domain=${d}`;
          }
        }
      }
    } catch {}
    listCookies();
    updateEstimate();
  }

  // CacheStorage
  async function listCaches() {
    el.caches.error.style.display = 'none';
    el.caches.table.innerHTML = '';
    if (!('caches' in window)) {
      el.caches.count.textContent = '(não suportado)';
      el.caches.error.style.display = '';
      el.caches.error.textContent = 'CacheStorage não suportado neste ambiente.';
      return;
    }
    try {
      const names = await caches.keys();
      names.sort();
      for (const name of names) {
        let entriesCount = 0;
        try {
          const cache = await caches.open(name);
          const reqs = await cache.keys();
          entriesCount = reqs.length;
        } catch {}
        const removeBtn = createActionButton('Remover', async () => {
          try { await caches.delete(name); } finally { listCaches(); updateEstimate(); }
        }, 'danger');
        el.caches.table.appendChild(renderTableRow([name, String(entriesCount), removeBtn]));
      }
      el.caches.count.textContent = `(${names.length})`;
    } catch (err) {
      el.caches.error.style.display = '';
      el.caches.error.textContent = 'Erro ao acessar CacheStorage: ' + (err && err.message ? err.message : String(err));
      el.caches.count.textContent = '(erro)';
    }
  }

  async function clearAllCaches() {
    if (!('caches' in window)) return;
    try {
      const names = await caches.keys();
      await Promise.all(names.map((n) => caches.delete(n)));
    } catch {}
    listCaches();
    updateEstimate();
  }

  // IndexedDB (melhor esforço)
  function indexedDBSupported() {
    return typeof indexedDB !== 'undefined';
  }

  function listIdb() {
    el.idb.error.style.display = 'none';
    el.idb.table.innerHTML = '';
    if (!indexedDBSupported()) {
      el.idb.count.textContent = '(não suportado)';
      el.idb.error.style.display = '';
      el.idb.error.textContent = 'IndexedDB não suportado neste ambiente.';
      return;
    }
    // Sem API universal para listar bancos. Vamos tentar heurísticas conhecidas:
    // 1) Enumerar via indexedDB.databases() quando disponível (não padrão em todos browsers)
    const anyIDB = indexedDB;
    if (typeof anyIDB.databases === 'function') {
      anyIDB.databases().then((dbs) => {
        renderIdbList(Array.isArray(dbs) ? dbs : []);
      }).catch((err) => {
        el.idb.error.style.display = '';
        el.idb.error.textContent = 'Erro ao listar IndexedDB: ' + (err && err.message ? err.message : String(err));
        el.idb.count.textContent = '(erro)';
      });
      return;
    }
    // 2) Fallback: não há como listar; exibir orientação
    const row = renderTableRow(['Indisponível', 'O navegador não expõe a listagem de bancos.', '']);
    el.idb.table.appendChild(row);
    el.idb.count.textContent = '(?)';
  }

  function renderIdbList(dbs) {
    el.idb.table.innerHTML = '';
    const sanitized = dbs.filter(Boolean);
    sanitized.sort((a, b) => String(a.name || '').localeCompare(String(b.name || '')));
    for (const db of sanitized) {
      const name = db.name || '(sem nome)';
      const info = `versão: ${db.version || '?'}${db.origin ? ` | origem: ${db.origin}` : ''}`;
      const removeBtn = createActionButton('Remover', () => {
        try { indexedDB.deleteDatabase(name); } catch {}
        // Pequena espera para o delete se propagar
        setTimeout(() => { listIdb(); updateEstimate(); }, 250);
      }, 'danger');
      el.idb.table.appendChild(renderTableRow([name, info, removeBtn]));
    }
    el.idb.count.textContent = `(${sanitized.length})`;
  }

  // Service Workers
  function listServiceWorkers() {
    el.sw.error.style.display = 'none';
    el.sw.table.innerHTML = '';
    if (!('serviceWorker' in navigator)) {
      el.sw.count.textContent = '(não suportado)';
      el.sw.error.style.display = '';
      el.sw.error.textContent = 'Service Workers não suportados neste ambiente.';
      return;
    }
    navigator.serviceWorker.getRegistrations().then((regs) => {
      regs.sort((a, b) => String(a.scope).localeCompare(String(b.scope)));
      for (const reg of regs) {
        const scope = reg.scope;
        const state = reg.active ? 'ativo' : reg.installing ? 'instalando' : reg.waiting ? 'aguardando' : 'desconhecido';
        const removeBtn = createActionButton('Desregistrar', () => {
          reg.unregister().finally(() => { listServiceWorkers(); updateEstimate(); });
        }, 'danger');
        el.sw.table.appendChild(renderTableRow([scope, state, removeBtn]));
      }
      el.sw.count.textContent = `(${regs.length})`;
    }).catch((err) => {
      el.sw.error.style.display = '';
      el.sw.error.textContent = 'Erro ao acessar Service Workers: ' + (err && err.message ? err.message : String(err));
      el.sw.count.textContent = '(erro)';
    });
  }

  // Estimativa de uso
  async function updateEstimate() {
    if (!navigator.storage || !navigator.storage.estimate) {
      storageSummaryEl.className = 'text-muted';
      storageSummaryEl.textContent = 'Estimativa não suportada neste ambiente.';
      return;
    }
    try {
      const { usage, quota } = await navigator.storage.estimate();
      const pct = quota ? Math.round((usage / quota) * 100) : 0;
      storageSummaryEl.className = 'text-muted';
      storageSummaryEl.textContent = `Uso estimado: ${formatBytes(usage)} de ${formatBytes(quota)} (${pct}%)`;
      const bar = document.getElementById('storageProgressBar');
      if (bar) {
        bar.style.width = `${Math.min(100, Math.max(0, pct))}%`;
        bar.setAttribute('aria-valuenow', String(pct));
      }
    } catch (err) {
      storageSummaryEl.className = 'text-danger';
      storageSummaryEl.textContent = 'Falha ao obter estimativa.';
    }
  }

  function formatBytes(bytes) {
    if (!bytes && bytes !== 0) return '?';
    const units = ['B','KB','MB','GB','TB'];
    let idx = 0;
    let value = bytes;
    while (value >= 1024 && idx < units.length - 1) { value /= 1024; idx++; }
    return `${value.toFixed(2)} ${units[idx]}`;
  }

  // Clear All (melhor esforço)
  async function clearAll() {
    disableAll(true);
    try {
      clearLocalStorage();
      clearSessionStorage();
      clearAllCookies();
      await clearAllCaches();
      // IndexedDB: se databases() disponível, deletar todas
      try {
        const anyIDB = indexedDB;
        if (typeof anyIDB.databases === 'function') {
          const dbs = await anyIDB.databases();
          if (Array.isArray(dbs)) {
            for (const db of dbs) {
              if (db && db.name) {
                try { indexedDB.deleteDatabase(db.name); } catch {}
              }
            }
          }
        }
      } catch {}
      // Service workers
      try {
        if ('serviceWorker' in navigator) {
          const regs = await navigator.serviceWorker.getRegistrations();
          await Promise.all(regs.map((r) => r.unregister().catch(() => {})));
        }
      } catch {}
    } finally {
      await refreshAll();
      disableAll(false);
    }
  }

  function disableAll(disabled) {
    el.local.clearBtn.disabled = disabled;
    el.session.clearBtn.disabled = disabled;
    el.cookies.clearBtn.disabled = disabled;
    el.caches.clearBtn.disabled = disabled;
    el.idb.clearBtn.disabled = disabled;
    el.sw.clearBtn.disabled = disabled;
    el.refreshBtn.disabled = disabled;
    el.reloadBtn.disabled = disabled;
    el.clearAllBtn.disabled = disabled;
  }

  async function refreshAll() {
    listLocalStorage();
    listSessionStorage();
    listCookies();
    await listCaches();
    listIdb();
    listServiceWorkers();
    renderDiagTelegram();
    await updateEstimate();
    setUpdatedNow();
  }

  // Eventos UI
  function wireEvents() {
    el.local.clearBtn.addEventListener('click', clearLocalStorage);
    el.session.clearBtn.addEventListener('click', clearSessionStorage);
    el.cookies.clearBtn.addEventListener('click', clearAllCookies);
    el.caches.clearBtn.addEventListener('click', () => { clearAllCaches(); });
    el.idb.clearBtn.addEventListener('click', async () => {
      try {
        const anyIDB = indexedDB;
        if (typeof anyIDB.databases === 'function') {
          const dbs = await anyIDB.databases();
          if (Array.isArray(dbs)) {
            for (const db of dbs) {
              if (db && db.name) {
                try { indexedDB.deleteDatabase(db.name); } catch {}
              }
            }
          }
        }
      } catch {}
      setTimeout(() => { listIdb(); updateEstimate(); }, 250);
    });
    el.sw.clearBtn.addEventListener('click', async () => {
      try {
        if ('serviceWorker' in navigator) {
          const regs = await navigator.serviceWorker.getRegistrations();
          await Promise.all(regs.map((r) => r.unregister().catch(() => {})));
        }
      } catch {}
      listServiceWorkers();
      updateEstimate();
    });
    el.refreshBtn.addEventListener('click', () => { refreshAll(); });
    el.reloadBtn.addEventListener('click', () => {
      const url = new URL(location.href);
      url.searchParams.set('_cb', String(Date.now()));
      location.replace(url.toString());
    });
    el.clearAllBtn.addEventListener('click', () => {
      try {
        const modalEl = document.getElementById('confirmClearAllModal');
        const modal = new bootstrap.Modal(modalEl);
        modal.show();
        el.confirmClearAllBtn.onclick = async () => {
          modal.hide();
          await clearAll();
        };
      } catch {
        // fallback se bootstrap não existir
        if (confirm('Confirmar: apagar TUDO?')) clearAll();
      }
    });

    // diag telegram actions
    const copy = (text) => {
      try {
        if (navigator.clipboard && navigator.clipboard.writeText) return navigator.clipboard.writeText(text);
        const ta = document.createElement('textarea'); ta.value = text; document.body.appendChild(ta); ta.select(); document.execCommand('copy'); document.body.removeChild(ta);
      } catch {}
    };
    if (diag.copyRawBtn) diag.copyRawBtn.addEventListener('click', () => copy(diag.raw ? diag.raw.textContent : ''));
    if (diag.copyJsonBtn) diag.copyJsonBtn.addEventListener('click', () => copy(diag.json ? diag.json.textContent : ''));
    if (diag.clearRawBtn) diag.clearRawBtn.addEventListener('click', () => {
      try { localStorage.removeItem('tg_webapp_hash_raw'); } catch {}
      renderDiagTelegram();
    });
  }

  function renderDiagTelegram() {
    try {
      const raw = localStorage.getItem('tg_webapp_hash_raw') || '';
      if (diag.raw) diag.raw.textContent = raw || '(vazio)';
      const decoded = raw.startsWith('#') ? raw.substring(1) : raw;
      const params = new URLSearchParams(decoded);
      const data = {}; for (const [k, v] of params.entries()) data[k] = v;
      if (diag.json) diag.json.textContent = JSON.stringify(data, null, 2);
    } catch (e) {
      if (diag.json) diag.json.textContent = 'Falha ao decodificar';
    }
  }

  // Init
  (function init() {
    setEnvInfo();
    wireEvents();
    refreshAll();
  })();
})();


